'use server'

import { getSheet, SHEET_TITLES, initializeSheets } from '@/lib/googleSheets'
import { revalidatePath } from 'next/cache'
import { z } from 'zod'

// Helper to generate simple numeric ID (timestamp-based for simplicity in this scale)
const generateId = () => Date.now();

// Schema definitions
const EntrySchema = z.object({
    date: z.string(),
    categoryId: z.coerce.number(),
    content: z.string().optional(),
    amount: z.coerce.number(),
    paymentMethodId: z.coerce.number().optional(),
    remarks: z.string().optional(),
})

const CategorySchema = z.object({ name: z.string().min(1) })
const PaymentMethodSchema = z.object({ name: z.string().min(1) })
const FixedCostSchema = z.object({
    name: z.string().min(1),
    amount: z.coerce.number().min(1),
    dayOfMonth: z.coerce.number().min(1).max(31),
    categoryId: z.coerce.number(),
    paymentMethodId: z.coerce.number().optional()
})

// Helper for safe parsing
const safeNumber = (val: any) => {
    if (typeof val === 'number') return val;
    if (!val) return 0;
    // Remove commas, currency symbols
    const clean = String(val).replace(/[¥,]/g, '');
    const num = Number(clean);
    return isNaN(num) ? 0 : num;
}

const safeDate = (val: any) => {
    const d = new Date(val);
    return isNaN(d.getTime()) ? new Date() : d; // Fallback to now if invalid
}

// --- Entries --- //

export async function addEntry(formData: FormData) {
    try {
        const rawData: any = {
            date: formData.get('date'),
            categoryId: formData.get('categoryId'),
            content: formData.get('content'),
            amount: formData.get('amount'),
            paymentMethodId: formData.get('paymentMethodId'),
            remarks: formData.get('remarks'),
        }
        if (rawData.paymentMethodId === '') rawData.paymentMethodId = undefined;
        const val = EntrySchema.parse(rawData)

        const sheet = await getSheet(SHEET_TITLES.ENTRIES);
        await sheet.addRow({
            id: generateId(),
            date: val.date, // Store as YYYY-MM-DD string
            content: val.content || '',
            amount: val.amount,
            categoryId: val.categoryId,
            paymentMethodId: val.paymentMethodId || '',
            remarks: val.remarks || '',
            createdAt: new Date().toISOString()
        });

        revalidatePath('/')
        return { success: true }
    } catch (e) {
        console.error(e);
        return { success: false, error: 'Failed to add entry. Check credentials.' }
    }
}

export async function updateEntry(id: number, formData: FormData) {
    try {
        const rawData: any = {
            date: formData.get('date'),
            categoryId: formData.get('categoryId'),
            content: formData.get('content'),
            amount: formData.get('amount'),
            paymentMethodId: formData.get('paymentMethodId'),
            remarks: formData.get('remarks'),
        }
        if (rawData.paymentMethodId === '') rawData.paymentMethodId = undefined;
        const val = EntrySchema.parse(rawData)

        const sheet = await getSheet(SHEET_TITLES.ENTRIES);
        const rows = await sheet.getRows();
        const row = rows.find(r => Number(r.get('id')) === id);

        if (row) {
            row.assign({
                date: val.date,
                content: val.content || '',
                amount: val.amount,
                categoryId: val.categoryId,
                paymentMethodId: val.paymentMethodId || '',
                remarks: val.remarks || '',
            });
            await row.save();
        }

        revalidatePath('/')
        return { success: true }
    } catch (e) {
        console.error(e);
        return { success: false, error: 'Failed' }
    }
}

export async function deleteEntry(id: number) {
    try {
        const sheet = await getSheet(SHEET_TITLES.ENTRIES);
        const rows = await sheet.getRows();
        const row = rows.find(r => Number(r.get('id')) === id);
        if (row) await row.delete();
        revalidatePath('/')
        return { success: true }
    } catch (e) {
        return { success: false }
    }
}

export async function getEntries() {
    try {
        const [entrySheet, catSheet, pmSheet] = await Promise.all([
            getSheet(SHEET_TITLES.ENTRIES),
            getSheet(SHEET_TITLES.CATEGORIES),
            getSheet(SHEET_TITLES.PAYMENT_METHODS)
        ]);

        const [entryRows, catRows, pmRows] = await Promise.all([
            entrySheet.getRows(),
            catSheet.getRows(),
            pmSheet.getRows()
        ]);

        const categories = catRows.reduce((acc: any, row) => ({ ...acc, [row.get('id')]: { id: Number(row.get('id')), name: row.get('name') } }), {});
        const pms = pmRows.reduce((acc: any, row) => ({ ...acc, [row.get('id')]: { id: Number(row.get('id')), name: row.get('name') } }), {});

        const entries = entryRows.map(row => {
            const rawDate = row.get('date');
            const date = isNaN(new Date(rawDate).getTime()) ? new Date() : new Date(rawDate);

            return {
                id: safeNumber(row.get('id')),
                date: date,
                content: row.get('content'),
                amount: safeNumber(row.get('amount')),
                categoryId: safeNumber(row.get('categoryId')),
                paymentMethodId: row.get('paymentMethodId') ? safeNumber(row.get('paymentMethodId')) : null,
                remarks: row.get('remarks'),
                createdAt: safeDate(row.get('createdAt') || new Date().toISOString()),
                category: categories[row.get('categoryId')] || { name: 'Unknown', type: 'expense', createdAt: new Date() },
                paymentMethod: pms[row.get('paymentMethodId')] || null
            };
        }).sort((a, b) => b.date.getTime() - a.date.getTime()).slice(0, 50);

        return entries;
    } catch (e) {
        console.error(e);
        return [];
    }
}

// --- Categories --- //

export async function getCategories() {
    try {
        const sheet = await getSheet(SHEET_TITLES.CATEGORIES);
        const rows = await sheet.getRows();
        return rows.map(row => ({
            id: Number(row.get('id')),
            name: row.get('name'),
            type: row.get('type') || 'expense',
            createdAt: new Date(row.get('createdAt') || new Date().toISOString())
        }));
    } catch (e) {
        return [];
    }
}

export async function addCategory(formData: FormData) {
    try {
        const name = formData.get('name');
        const validated = CategorySchema.parse({ name });
        const sheet = await getSheet(SHEET_TITLES.CATEGORIES);
        await sheet.addRow({
            id: generateId(),
            name: validated.name,
            type: 'expense',
            createdAt: new Date().toISOString()
        });
        revalidatePath('/settings');
        revalidatePath('/');
        return { success: true };
    } catch (e) {
        return { success: false, error: 'Failed to add category' };
    }
}

export async function updateCategory(id: number, name: string) {
    try {
        const validated = CategorySchema.parse({ name });
        const sheet = await getSheet(SHEET_TITLES.CATEGORIES);
        const rows = await sheet.getRows();
        const row = rows.find(r => Number(r.get('id')) === id);
        if (row) {
            row.assign({ name: validated.name });
            await row.save();
        }
        revalidatePath('/settings');
        revalidatePath('/');
        return { success: true };
    } catch (e) {
        return { success: false, error: 'Failed' };
    }
}

export async function deleteCategory(id: number) {
    try {
        const sheet = await getSheet(SHEET_TITLES.CATEGORIES);
        const rows = await sheet.getRows();
        const row = rows.find(r => Number(r.get('id')) === id);
        if (row) await row.delete();
        revalidatePath('/settings');
        revalidatePath('/');
        return { success: true };
    } catch (e) {
        return { success: false, error: 'Failed' };
    }
}

// --- Payment Methods --- //

export async function getPaymentMethods() {
    try {
        const sheet = await getSheet(SHEET_TITLES.PAYMENT_METHODS);
        const rows = await sheet.getRows();
        return rows.map(row => ({
            id: Number(row.get('id')),
            name: row.get('name'),
            createdAt: new Date(row.get('createdAt') || new Date().toISOString())
        }));
    } catch (e) {
        return [];
    }
}

export async function addPaymentMethod(formData: FormData) {
    try {
        const name = formData.get('name');
        const validated = PaymentMethodSchema.parse({ name });
        const sheet = await getSheet(SHEET_TITLES.PAYMENT_METHODS);
        await sheet.addRow({
            id: generateId(),
            name: validated.name,
            createdAt: new Date().toISOString()
        });
        revalidatePath('/settings');
        revalidatePath('/');
        return { success: true };
    } catch (e) {
        return { success: false, error: 'Failed' };
    }
}

export async function updatePaymentMethod(id: number, name: string) {
    try {
        const validated = PaymentMethodSchema.parse({ name });
        const sheet = await getSheet(SHEET_TITLES.PAYMENT_METHODS);
        const rows = await sheet.getRows();
        const row = rows.find(r => Number(r.get('id')) === id);
        if (row) {
            row.assign({ name: validated.name });
            await row.save();
        }
        revalidatePath('/settings');
        revalidatePath('/');
        return { success: true };
    } catch (e) {
        return { success: false, error: 'Failed' };
    }
}

export async function deletePaymentMethod(id: number) {
    try {
        const sheet = await getSheet(SHEET_TITLES.PAYMENT_METHODS);
        const rows = await sheet.getRows();
        const row = rows.find(r => Number(r.get('id')) === id);
        if (row) await row.delete(); // Note: This doesn't check usage logic like Prisma did, simplifying for migration.
        revalidatePath('/settings');
        revalidatePath('/');
        return { success: true };
    } catch (e) {
        return { success: false, error: 'Failed' };
    }
}

// --- Budget --- //

export async function getBudget(month: string) {
    try {
        const sheet = await getSheet(SHEET_TITLES.BUDGET);
        const rows = await sheet.getRows();
        const row = rows.find(r => r.get('month') === month);
        if (row) return { month: row.get('month'), amount: safeNumber(row.get('amount')) };
        return null;
    } catch (e) {
        return null;
    }
}

export async function setBudget(month: string, amount: number) {
    try {
        const sheet = await getSheet(SHEET_TITLES.BUDGET);
        const rows = await sheet.getRows();
        const row = rows.find(r => r.get('month') === month);
        if (row) {
            row.assign({ amount, updatedAt: new Date().toISOString() });
            await row.save();
        } else {
            await sheet.addRow({
                month,
                amount,
                updatedAt: new Date().toISOString()
            });
        }
        revalidatePath('/');
        revalidatePath('/settings');
        return { success: true };
    } catch (e) {
        return { success: false, error: 'Failed' };
    }
}

export async function getMonthlyTotal(month: string) {
    try {
        const sheet = await getSheet(SHEET_TITLES.ENTRIES);
        const rows = await sheet.getRows();
        // Filter by YYYY-MM
        let total = 0;
        let variable = 0;
        let fixed = 0;

        rows.forEach(row => {
            if (row.get('date') && String(row.get('date')).startsWith(month)) {
                const amount = safeNumber(row.get('amount'));
                const content = row.get('content') || '';
                total += amount;
                if (content.startsWith('[固定費]')) {
                    fixed += amount;
                } else {
                    variable += amount;
                }
            }
        });
        return { total, variable, fixed };
    } catch (e) {
        return { total: 0, variable: 0, fixed: 0 };
    }
}

// --- Fixed Costs --- //

export async function getFixedCosts() {
    try {
        const [fcSheet, catSheet, pmSheet] = await Promise.all([
            getSheet(SHEET_TITLES.FIXED_COSTS),
            getSheet(SHEET_TITLES.CATEGORIES),
            getSheet(SHEET_TITLES.PAYMENT_METHODS)
        ]);
        const [fcRows, catRows, pmRows] = await Promise.all([
            fcSheet.getRows(),
            catSheet.getRows(),
            pmSheet.getRows()
        ]);

        // Create lookup maps
        const categories: any = {};
        catRows.forEach(r => categories[r.get('id')] = { id: Number(r.get('id')), name: r.get('name') });

        const pms: any = {};
        pmRows.forEach(r => pms[r.get('id')] = { id: Number(r.get('id')), name: r.get('name') });

        return fcRows.map(row => ({
            id: safeNumber(row.get('id')),
            name: row.get('name'),
            amount: safeNumber(row.get('amount')),
            dayOfMonth: safeNumber(row.get('dayOfMonth')),
            categoryId: safeNumber(row.get('categoryId')),
            paymentMethodId: row.get('paymentMethodId') ? safeNumber(row.get('paymentMethodId')) : null,
            category: categories[row.get('categoryId')] || { name: 'Unknown' },
            paymentMethod: pms[row.get('paymentMethodId')] || null
        }));
    } catch (e) {
        return [];
    }
}

export async function addFixedCost(formData: FormData) {
    try {
        const raw: any = {
            name: formData.get('name'),
            amount: formData.get('amount'),
            dayOfMonth: formData.get('dayOfMonth'),
            categoryId: formData.get('categoryId'),
            paymentMethodId: formData.get('paymentMethodId')
        }
        if (raw.paymentMethodId === '') raw.paymentMethodId = undefined
        const val = FixedCostSchema.parse(raw)

        const sheet = await getSheet(SHEET_TITLES.FIXED_COSTS);
        await sheet.addRow({
            id: generateId(),
            name: val.name,
            amount: val.amount,
            dayOfMonth: val.dayOfMonth,
            categoryId: val.categoryId,
            paymentMethodId: val.paymentMethodId || '',
            createdAt: new Date().toISOString()
        });
        revalidatePath('/settings');
        return { success: true };
    } catch (e) {
        return { success: false, error: 'Failed' };
    }
}

export async function deleteFixedCost(id: number) {
    try {
        const sheet = await getSheet(SHEET_TITLES.FIXED_COSTS);
        const rows = await sheet.getRows();
        const row = rows.find(r => Number(r.get('id')) === id);
        if (row) await row.delete();
        revalidatePath('/settings');
        return { success: true };
    } catch (e) {
        return { success: false };
    }
}

export async function applyFixedCosts(month: string) {
    try {
        const fixedCosts = await getFixedCosts();
        const [year, m] = month.split('-').map(Number);
        const sheet = await getSheet(SHEET_TITLES.ENTRIES);

        let count = 0;
        for (const fc of fixedCosts) {
            const date = new Date(year, m - 1, fc.dayOfMonth);
            // Adjust if month rolled over due to day overflow (javascript date behavior)
            if (date.getMonth() !== m - 1) {
                date.setDate(0) // Roll back to last day of previous month
            }
            const dateStr = date.toISOString().split('T')[0];

            await sheet.addRow({
                id: generateId(), // Note: loops might generate same ID if fast? Date.now() is ms precision. Add random or sleep?
                // simple workaround:
                // id: generateId() + count, 
                date: dateStr,
                content: `[固定費] ${fc.name}`,
                amount: fc.amount,
                categoryId: fc.categoryId,
                paymentMethodId: fc.paymentMethodId || '',
                remarks: '',
                createdAt: new Date().toISOString()
            });
            count++;
        }
        revalidatePath('/');
        return { success: true, count };
    } catch (e) {
        return { success: false };
    }
}

// Initial seed helper
export async function seedDefaults() {
    try {
        await initializeSheets();

        // Check categories
        const catSheet = await getSheet(SHEET_TITLES.CATEGORIES);
        const catRows = await catSheet.getRows();
        if (catRows.length === 0) {
            const defaults = [
                { name: '食費', type: 'expense' },
                { name: '日用品', type: 'expense' },
                { name: '交通費', type: 'expense' },
                { name: '交際費', type: 'expense' },
                { name: '趣味', type: 'expense' },
                { name: '固定費', type: 'expense' },
                { name: '入金', type: 'income' },
            ];
            for (const d of defaults) {
                await catSheet.addRow({ id: generateId() + Math.floor(Math.random() * 1000), ...d, createdAt: new Date().toISOString() });
            }
        }

        // Check PMs
        const pmSheet = await getSheet(SHEET_TITLES.PAYMENT_METHODS);
        const pmRows = await pmSheet.getRows();
        if (pmRows.length === 0) {
            const defaults = [
                { name: '現金' },
                { name: 'クレジットカード' },
                { name: '銀行振込' },
                { name: 'PayPay' },
            ];
            for (const d of defaults) {
                await pmSheet.addRow({ id: generateId() + Math.floor(Math.random() * 1000), ...d, createdAt: new Date().toISOString() });
            }
        }
    } catch (e) {
        console.error('Seeding failed (likely no credentials yet).', e);
    }
}
