'use client'

import { useRef } from 'react'
import { addEntry } from '@/app/actions'

type Category = { id: number; name: string }
type PaymentMethod = { id: number; name: string }

export default function EntryForm({
    categories,
    paymentMethods,
}: {
    categories: Category[]
    paymentMethods: PaymentMethod[]
}) {
    const formRef = useRef<HTMLFormElement>(null)
    const today = new Date().toISOString().split('T')[0]

    async function action(formData: FormData) {
        const result = await addEntry(formData)
        if (result.success) {
            formRef.current?.reset()
            // Reset date to today as reset() clears it ?? No, reset() restores default values if defined, or clears if not.
            // Better to manually restore default date if needed, or rely on defaultValue.
        }
    }

    return (
        <div className="card">
            <h2 className="label" style={{ fontSize: '1.25rem', marginBottom: '1rem' }}>新しい入力</h2>
            <form ref={formRef} action={action}>
                <div className="input-group">
                    <label className="label" htmlFor="date">日付</label>
                    <input
                        type="date"
                        id="date"
                        name="date"
                        defaultValue={today}
                        className="input"
                        required
                    />
                </div>

                <div className="input-group">
                    <label className="label" htmlFor="amount">金額</label>
                    <input
                        type="number"
                        id="amount"
                        name="amount"
                        placeholder="¥0"
                        className="input"
                        style={{ fontSize: '1.5rem', fontWeight: 'bold' }}
                        required
                        inputMode="numeric"
                    />
                </div>

                <div className="input-group">
                    <label className="label" htmlFor="categoryId">カテゴリー</label>
                    <select id="categoryId" name="categoryId" className="select" required defaultValue="">
                        <option value="" disabled>選択してください</option>
                        {categories.map((c) => (
                            <option key={c.id} value={c.id}>
                                {c.name}
                            </option>
                        ))}
                    </select>
                </div>

                <div className="input-group">
                    <label className="label" htmlFor="content">内容（任意）</label>
                    <input
                        type="text"
                        id="content"
                        name="content"
                        className="input"
                        placeholder="例: ランチ、スーパー"
                    />
                </div>

                <div className="input-group">
                    <label className="label" htmlFor="paymentMethodId">支払方法</label>
                    <select id="paymentMethodId" name="paymentMethodId" className="select" defaultValue="">
                        <option value="" disabled>選択してください</option>
                        {paymentMethods.map((pm) => (
                            <option key={pm.id} value={pm.id}>
                                {pm.name}
                            </option>
                        ))}
                    </select>
                </div>

                <div className="input-group">
                    <label className="label" htmlFor="remarks">備考（任意）</label>
                    <textarea
                        id="remarks"
                        name="remarks"
                        className="textarea"
                        rows={2}
                    ></textarea>
                </div>

                <button type="submit" className="btn" style={{ width: '100%', padding: '0.75rem' }}>
                    登録する
                </button>
            </form>
        </div>
    )
}
