'use client'

import { useState } from 'react'
import { deleteEntry, updateEntry } from '@/app/actions'

// Basic types
type Category = {
    id: number
    name: string
    type: string
    createdAt: Date
}

type PaymentMethod = {
    id: number
    name: string
    createdAt: Date
}

type Entry = {
    id: number
    date: Date
    content: string | null
    amount: number
    remarks: string | null
    categoryId: number
    paymentMethodId: number | null
    createdAt: Date
    category: Category
    paymentMethod: PaymentMethod | null
}

type HistoryListProps = {
    entries: Entry[]
    categories: Category[]
    paymentMethods: PaymentMethod[]
}

export default function HistoryList({ entries, categories, paymentMethods }: HistoryListProps) {
    const [editingEntry, setEditingEntry] = useState<Entry | null>(null)

    async function handleUpdate(formData: FormData) {
        if (!editingEntry) return
        const res = await updateEntry(editingEntry.id, formData)
        if (res.success) {
            setEditingEntry(null)
        } else {
            alert('Update failed')
        }
    }

    return (
        <div className="card" style={{ marginTop: '1.5rem' }}>
            <h3 className="label">最近の履歴</h3>
            {entries.length === 0 ? (
                <p style={{ padding: '1rem', textAlign: 'center', color: 'var(--text-muted)' }}>データがありません</p>
            ) : (
                <ul style={{ listStyle: 'none' }}>
                    {entries.map((entry) => (
                        <li key={entry.id} style={{ padding: '0.75rem 0', borderBottom: '1px solid var(--border-color)' }}>
                            {editingEntry?.id === entry.id ? (
                                <form action={handleUpdate} style={{ display: 'flex', flexDirection: 'column', gap: '0.5rem' }}>
                                    <div style={{ display: 'flex', gap: '0.5rem' }}>
                                        <input name="date" type="date" className="input" defaultValue={new Date(entry.date).toISOString().split('T')[0]} required />
                                        <input name="amount" type="number" className="input" defaultValue={entry.amount} required />
                                    </div>
                                    <div style={{ display: 'flex', gap: '0.5rem' }}>
                                        <select name="categoryId" className="select" defaultValue={entry.categoryId}>
                                            {categories.map(c => <option key={c.id} value={c.id}>{c.name}</option>)}
                                        </select>
                                        <select name="paymentMethodId" className="select" defaultValue={entry.paymentMethodId ?? ''}>
                                            <option value="">支払方法なし</option>
                                            {paymentMethods.map(p => <option key={p.id} value={p.id}>{p.name}</option>)}
                                        </select>
                                    </div>
                                    <input name="content" className="input" defaultValue={entry.content || ''} placeholder="内容" />
                                    <div style={{ display: 'flex', gap: '0.5rem', justifyContent: 'flex-end' }}>
                                        <button type="button" onClick={() => setEditingEntry(null)} className="btn-outline" style={{ fontSize: '0.8rem', padding: '0.25rem 0.5rem' }}>キャンセル</button>
                                        <button type="submit" className="btn" style={{ fontSize: '0.8rem', padding: '0.25rem 0.5rem' }}>保存</button>
                                    </div>
                                </form>
                            ) : (
                                <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                                    <div>
                                        <div style={{ fontSize: '0.875rem', fontWeight: 'bold' }}>{entry.category.name}</div>
                                        <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)' }}>
                                            {new Date(entry.date).toLocaleDateString('ja-JP', { month: 'numeric', day: 'numeric' })}
                                            {entry.content && ` · ${entry.content}`}
                                            {entry.paymentMethod && ` · ${entry.paymentMethod.name}`}
                                        </div>
                                    </div>
                                    <div style={{ display: 'flex', alignItems: 'center', gap: '1rem' }}>
                                        <div style={{ fontWeight: 'bold' }}>
                                            ¥{entry.amount.toLocaleString()}
                                        </div>
                                        <div style={{ display: 'flex', gap: '0.25rem' }}>
                                            <button
                                                onClick={() => setEditingEntry(entry)}
                                                style={{ border: 'none', background: 'none', cursor: 'pointer', color: 'var(--primary-color)', fontSize: '0.8rem', padding: '0.25rem' }}
                                            >
                                                編集
                                            </button>
                                            <button
                                                onClick={() => {
                                                    if (confirm('削除しますか？')) deleteEntry(entry.id)
                                                }}
                                                style={{ border: 'none', background: 'none', cursor: 'pointer', color: 'var(--danger-color)', fontSize: '0.8rem', padding: '0.25rem' }}
                                            >
                                                削除
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            )}
                        </li>
                    ))}
                </ul>
            )}
        </div>
    )
}
