import { GoogleSpreadsheet } from 'google-spreadsheet';
import { JWT } from 'google-auth-library';

// Environment variables
const SERVICE_ACCOUNT_EMAIL = process.env.GOOGLE_SERVICE_ACCOUNT_EMAIL;
const PRIVATE_KEY = process.env.GOOGLE_PRIVATE_KEY?.replace(/\\n/g, '\n'); // Handle newlines in env var
const SHEET_ID = process.env.GOOGLE_SHEET_ID;

if (!SERVICE_ACCOUNT_EMAIL || !PRIVATE_KEY || !SHEET_ID) {
    console.warn('Google Sheets credentials are missing. Please check .env file.');
}

const serviceAccountAuth = new JWT({
    email: SERVICE_ACCOUNT_EMAIL,
    key: PRIVATE_KEY,
    scopes: ['https://www.googleapis.com/auth/spreadsheets'],
});

export const doc = new GoogleSpreadsheet(SHEET_ID as string, serviceAccountAuth);

export const SHEET_TITLES = {
    ENTRIES: 'Entries',
    CATEGORIES: 'Categories',
    PAYMENT_METHODS: 'PaymentMethods',
    FIXED_COSTS: 'FixedCosts',
    BUDGET: 'Budget',
};

// Headers configuration for each sheet
export const HEADERS = {
    ENTRIES: ['id', 'date', 'content', 'amount', 'categoryId', 'paymentMethodId', 'remarks', 'createdAt'],
    CATEGORIES: ['id', 'name', 'type', 'createdAt'],
    PAYMENT_METHODS: ['id', 'name', 'createdAt'],
    FIXED_COSTS: ['id', 'name', 'amount', 'dayOfMonth', 'categoryId', 'paymentMethodId', 'createdAt'],
    BUDGET: ['month', 'amount', 'updatedAt'],
};

export async function getSheet(title: string) {
    await doc.loadInfo();
    let sheet = doc.sheetsByTitle[title];

    // Find the header config for this title
    const key = Object.keys(SHEET_TITLES).find(k => SHEET_TITLES[k as keyof typeof SHEET_TITLES] === title);
    const headers = key ? HEADERS[key as keyof typeof HEADERS] : [];

    if (!sheet) {
        sheet = await doc.addSheet({ title });
        if (headers.length > 0) await sheet.setHeaderRow(headers);
    } else {
        // Check if headers are present. If getRows() is called on an empty sheet (no headers), it throws.
        // We try to load headers, and if it fails (empty), we set them.
        try {
            await sheet.loadHeaderRow();
        } catch (e) {
            // Likely "No values in the header row"
            if (headers.length > 0) {
                await sheet.setHeaderRow(headers);
            }
        }
    }
    return sheet;
}

export async function initializeSheets() {
    if (!SERVICE_ACCOUNT_EMAIL) throw new Error('Missing Google Credentials');
    await doc.loadInfo();

    for (const [key, title] of Object.entries(SHEET_TITLES)) {
        let sheet = doc.sheetsByTitle[title];
        const headers = HEADERS[key as keyof typeof HEADERS] || [];

        if (!sheet) {
            sheet = await doc.addSheet({ title });
            if (headers.length > 0) await sheet.setHeaderRow(headers);
        } else {
            // Ensure headers exist for existing sheets too
            try {
                await sheet.loadHeaderRow();
            } catch (e) {
                if (headers.length > 0) await sheet.setHeaderRow(headers);
            }
        }
    }
}
